importScripts(
    "https://storage.googleapis.com/workbox-cdn/releases/6.5.4/workbox-sw.js"
);

const CACHE_NAME = "mi-pwa-cache-v5"; // Cambia la versión cada vez que actualices algo
const STATIC_RESOURCES = [
    { url: "/", revision: "2" },
    { url: "/images/favicon.ico", revision: "2" },
    { url: "/images/progressive/android-chrome-192x192.png", revision: "2" },
    { url: "/images/progressive/android-chrome-512x512.png", revision: "2" },
];

// Precaching de recursos estáticos
workbox.precaching.precacheAndRoute(STATIC_RESOURCES);

// Estrategia de caché para imágenes (CacheFirst)
workbox.routing.registerRoute(
    ({ request }) => request.destination === "image",
    new workbox.strategies.CacheFirst({
        cacheName: "images-cache",
        plugins: [
            new workbox.cacheableResponse.CacheableResponsePlugin({
                statuses: [0, 200],
            }),
            new workbox.expiration.ExpirationPlugin({
                maxEntries: 30, // Reduce el número de imágenes almacenadas
                maxAgeSeconds: 15 * 24 * 60 * 60, // Expiran en 15 días en vez de 30
            }),
        ],
    })
);

// Estrategia de caché para fuentes (CacheFirst)
workbox.routing.registerRoute(
    ({ request }) => request.destination === "font",
    new workbox.strategies.CacheFirst({
        cacheName: "fonts-cache",
        plugins: [
            new workbox.cacheableResponse.CacheableResponsePlugin({
                statuses: [0, 200],
            }),
        ],
    })
);

// Estrategia de caché para archivos JS y CSS (NetworkFirst)
workbox.routing.registerRoute(
    ({ request }) =>
        request.destination === "script" || request.destination === "style",
    new workbox.strategies.NetworkFirst({
        cacheName: "other-resources-cache",
        networkTimeoutSeconds: 5,
        plugins: [
            new workbox.cacheableResponse.CacheableResponsePlugin({
                statuses: [0, 200],
            }),
        ],
    })
);

// Manejo de instalación y activación
self.addEventListener("install", (event) => {
    console.log("📌 Service Worker instalado");
    self.skipWaiting(); // Permite activar la nueva versión inmediatamente
});

self.addEventListener("activate", (event) => {
    console.log("♻️ Service Worker activado");
    event.waitUntil(
        caches.keys().then((cacheNames) => {
            return Promise.all(
                cacheNames
                    .filter(
                        (cacheName) =>
                            cacheName.startsWith("mi-pwa-cache-") &&
                            cacheName !== CACHE_NAME
                    )
                    .map((cacheName) => caches.delete(cacheName))
            );
        })
    );
    self.clients.claim(); // Asegura que la nueva versión se active en todos los clientes
});

// Manejo de notificaciones push
self.addEventListener("push", (event) => {
    console.log("📩 Notificación push recibida");
    const options = {
        body: event.data ? event.data.text() : "Nueva notificación",
        icon: "/images/icon.png",
        badge: "/images/badge.png",
    };
    event.waitUntil(
        self.registration.showNotification("Notificación", options)
    );
});

// Manejo de clics en las notificaciones
self.addEventListener("notificationclick", (event) => {
    event.notification.close();
    if (event.action === "permitir") {
        console.log("✅ Usuario permitió las notificaciones");
    } else if (event.action === "rechazar") {
        console.log("❌ Usuario rechazó las notificaciones");
    }
});

// Intercepta las solicitudes para registrar en consola
self.addEventListener("fetch", (event) => {
    console.log("🔍 Interceptando solicitud:", event.request.url);
});
